function [D,r]=Create_Diffusion_Radial(num_r,num_theta,varargin)

%num_r, num_theta must be even
%varargin is the optional index mapping from u(i,j) to x(i) state vector

%Create the dimensionless discretized diffusion operator for radial coords
%r --> r / R, delta_r = 1 / num_r
%delta_theta = 2*pi / num_theta

%BCs:
%no flux at the r=1 boundary

%Treating the singular pole at r=0: the discretized points are offset 
%such that the first radial point (i=1) is located at r = 1 / num_r / 2.
%Radial flux across the pole is then in terms of i and
%mod((i-1)+num_r/2,num_r)+1

if (mod(num_r,2) > 0 || mod(num_theta,2) > 0)
    disp('Number of radial and degree increments must both be even.');
    D = [];
    return
end

if (nargin >= 3)
    index = varargin{1};
else
    index = @(i,j) (i-1)*num_theta + j; %an index that maps u(i,j) coordinates to x(i) state vector
end

D = sparse(num_r*num_theta,num_r*num_theta);

delta_th = 2*pi ./ num_theta;
delta_th_sq = delta_th^2;

delta_r = ones(num_r,1) ./ num_r;
r = delta_r(1) .* (0:num_r-1) + delta_r(1)/2; %first radial position is delta_r/2, then increments by delta_r
r_sq = r.^2;
delta_r_sq = delta_r.^2;

%First, handle the i = 1 special case where radial flux goes across the
%pole towards the opposing coordinate (equivalent to no-flux BC at r =
%delta_r
i=1;
j=1;
%this reads: the uij'th row and ui+1,j'th column of the diffusion matrix is ...
D(index(i,j),index(i+1,j)) = + 2 / delta_r_sq(i);
D(index(i,j),index(i,j)) = -2 / delta_r_sq(i) - 2 / delta_th_sq / r_sq(i);
D(index(i,j),index(i,j+1)) = + 1 / delta_th_sq / r_sq(i);
D(index(i,j),index(i,num_theta)) = + 1 / delta_th_sq / r_sq(i);

i=1;
for j=2:num_theta-1
        D(index(i,j),index(i+1,j)) = + 2 / delta_r_sq(i);
        D(index(i,j),index(i,j)) = -2 / delta_r_sq(i) - 2 / delta_th_sq / r_sq(i);
        D(index(i,j),index(i,j+1)) = + 1 / delta_th_sq / r_sq(i);
        D(index(i,j),index(i,j-1)) = + 1 / delta_th_sq / r_sq(i);
end

i=1;
j=num_theta;
D(index(i,j),index(i+1,j)) = + 2 / delta_r_sq(i);
D(index(i,j),index(i,j)) = -2 / delta_r_sq(i) - 2 / delta_th_sq / r_sq(i);
D(index(i,j),index(i,j-1)) = + 1 / delta_th_sq / r_sq(i);
D(index(i,j),index(i,1)) = + 1 / delta_th_sq / r_sq(i);

%Second, handle the normal discretizations without BCs
%Also, when j = 1, then j-1 is num_theta, and when j = num_theta, then j+1
%is 1

for i=2:num_r-1
    j=1;
    D(index(i,j),index(i+1,j)) = + 1 / r(i) / delta_r(i) / 2 + 1 / delta_r_sq(i);
    D(index(i,j),index(i-1,j)) = - 1 / r(i) / delta_r(i) / 2 + 1 / delta_r_sq(i);
    D(index(i,j),index(i,j)) = - 2 / delta_r_sq(i)  - 2 / delta_th_sq / r_sq(i);
    D(index(i,j),index(i,j+1)) = + 1 / delta_th_sq / r_sq(i);
    D(index(i,j),index(i,num_theta)) = + 1 / delta_th_sq / r_sq(i);

    for j=2:num_theta-1
        D(index(i,j),index(i+1,j)) = + 1 / r(i) / delta_r(i) / 2 + 1 / delta_r_sq(i);
        D(index(i,j),index(i-1,j)) = - 1 / r(i) / delta_r(i) / 2 + 1 / delta_r_sq(i);
        D(index(i,j),index(i,j)) = - 2 / delta_r_sq(i)  - 2 / delta_th_sq / r_sq(i);
        D(index(i,j),index(i,j+1)) = + 1 / delta_th_sq / r_sq(i);
        D(index(i,j),index(i,j-1)) = + 1 / delta_th_sq / r_sq(i);
    end
    
    j=num_theta;
    D(index(i,j),index(i+1,j)) = + 1 / r(i) / delta_r(i) / 2 + 1 / delta_r_sq(i);
    D(index(i,j),index(i-1,j)) = - 1 / r(i) / delta_r(i) / 2 + 1 / delta_r_sq(i);
    D(index(i,j),index(i,j)) = - 2 / delta_r_sq(i)  - 2 / delta_th_sq / r_sq(i);
    D(index(i,j),index(i,1)) = + 1 / delta_th_sq / r_sq(i);
    D(index(i,j),index(i,j-1)) = + 1 / delta_th_sq / r_sq(i);
    
end

%Finally, handle the BCs at i = num_r

%with j = 1, j-1 is num_theta
i=num_r;
j=1;
D(index(i,j),index(i,j)) = +1 / r(i) / delta_r(i) / 2 - 1 / delta_r_sq(i) - 2 / delta_th_sq / r_sq(i);
D(index(i,j),index(i-1,j)) = -1 / r(i) / delta_r(i) / 2 + 1 / delta_r_sq(i);
D(index(i,j),index(i,num_theta)) = + 1 / delta_th_sq / r_sq(i);
D(index(i,j),index(i,j+1)) = + 1 / delta_th_sq / r_sq(i);

%with j = num_theta, j+1 is 1
i=num_r;
j=num_theta;
D(index(i,j),index(i,j)) = +1 / r(i) / delta_r(i) / 2 - 1 / delta_r_sq(i) - 2 / delta_th_sq / r_sq(i);
D(index(i,j),index(i-1,j)) = -1 / r(i) / delta_r(i) / 2 + 1 / delta_r_sq(i);
D(index(i,j),index(i,j-1)) = + 1 / delta_th_sq / r_sq(i);
D(index(i,j),index(i,1)) = + 1 / delta_th_sq / r_sq(i);

i=num_r;
for j=2:num_theta-1
    D(index(i,j),index(i,j)) = +1 / r(i) / delta_r(i) / 2 - 1 / delta_r_sq(i) - 2 / delta_th_sq / r_sq(i);
    D(index(i,j),index(i-1,j)) = -1 / r(i) / delta_r(i) / 2 + 1 / delta_r_sq(i);
    D(index(i,j),index(i,j-1)) = + 1 / delta_th_sq / r_sq(i);
    D(index(i,j),index(i,j+1)) = + 1 / delta_th_sq / r_sq(i);
end

%That should be it.
return